#coding: utf-8
import os,sys,json,time,hashlib

panelPath = os.getenv('BT_PANEL')
if not panelPath:
    panelPath = "/www/server/panel"

class FileSession:
    def __init__(self, session_expiry=3600):
        """
        初始化文件 Session
        :param session_dir: 存储 Session 的目录
        :param session_expiry: Session 过期时间（秒）
        """
        self.session_dir = os.path.join(panelPath, 'sessions')
        self.session_expiry = session_expiry
        self._cache = {}

        # 确保 Session 目录存在
        if not os.path.exists(self.session_dir):
            os.makedirs(self.session_dir)

    def _get_session_file(self, session_id):
        """
        获取 Session 文件路径
        :param session_id: Session ID
        :return: 文件路径
        """
        hashed_id = hashlib.md5(session_id.encode()).hexdigest()
        return os.path.join(self.session_dir, f"{hashed_id}.json")

    def __getitem__(self, session_id):
        """
        支持通过字典方式访问 Session
        :param session_id: Session ID
        :return: 存储的数据或 None
        """
        return self.get_session(session_id)

    def __setitem__(self, session_id, value):
        """
        支持通过字典方式设置 Session
        :param session_id: Session ID
        :param value: 存储的数据
        """
        self.create_session(session_id, value)

    def __delitem__(self, session_id):
        """
        支持通过字典方式删除 Session
        :param session_id: Session ID
        """
        self.delete_session(session_id)

    def __getattr__(self, name):
        """
        支持通过属性方式访问 Session
        :param name: 属性名
        :return: 存储的数据或 None
        """
        if name in self._cache:
            return self._cache[name]
        session_data = self.get_session(name)
        if session_data:
            self._cache[name] = session_data
            return session_data
        return None

    def __contains__(self, session_id):
        """
        支持使用 `in` 检查 Session 是否存在
        :param session_id: Session ID
        :return: True 或 False
        """
        return self.get_session(session_id) is not None

    def create_session(self, session_id, data):
        """
        创建新的 Session
        :param session_id: Session ID
        :param data: 存储的数据
        """
        session_file = self._get_session_file(session_id)
        session_data = {
            "data": data,
            "expiry": time.time() + self.session_expiry
        }
        with open(session_file, 'w') as f:
            json.dump(session_data, f)

    def get_session(self, session_id):
        """
        获取 Session 数据
        :param session_id: Session ID
        :return: 存储的数据或 None（如果 Session 不存在或已过期）
        """
        session_file = self._get_session_file(session_id)
        if not os.path.exists(session_file):
            return None

        with open(session_file, 'r') as f:
            session_data = json.load(f)

        # 检查是否过期
        if time.time() > session_data.get("expiry", 0):
            os.remove(session_file)
            return None

        return session_data.get("data")

    def delete_session(self, session_id):
        """
        删除 Session
        :param session_id: Session ID
        """
        session_file = self._get_session_file(session_id)
        if os.path.exists(session_file):
            os.remove(session_file)

    def clear_expired_sessions(self):
        """
        清理过期的 Session
        """
        for session_file in os.listdir(self.session_dir):
            session_path = os.path.join(self.session_dir, session_file)
            with open(session_path, 'r') as f:
                session_data = json.load(f)

            if time.time() > session_data.get("expiry", 0):
                os.remove(session_path)